/*
 * pet2ply is a utility for combining pet files and bitmap files into ply files
 * Copyright (C) 1999 Bryan Franklin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *u

/* if you would like to compile on a Unix machine instead
 * of a windows machine simply comment out the WIN32 line
 * and uncomment the UNIX line.
 */
#define WIN32
/*#define UNIX*/

/* If don't want pet2ply to ever ask you a question
 * then comment out the following line and recompile.
 * This will not only make pet2ply a lot smaller, but
 * will make it more suitable for the back end of a web-page.
 */
#define INTERACTIVE

/* If you would rather not create an HTML file for the ply files
 * them comment out the following line and recompile
 */
#define MAKEHTML

/* Everything needs config file, right? */
#ifdef WIN32
#define CONFIGFILE      "config.ini"
#elif defined(UNIX)
#define CONFIGFILE      ".pet2plyrc"
#endif

#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#ifdef WIN32
#include <io.h>
#endif
#ifdef UNIX
#include <unistd.h>
#define O_BINARY 0
#define S_IREAD S_IRUSR
#define S_IWRITE S_IWUSR
#endif

#define VERSION "1.0"

/* EXIT
 * 	Print [Hit enter to continue] and exit with appropriate return code
 * Input
 * 	code : return code to send back to the OS
 * Returns
 * 	Nothing, this function always terminates the program
 */
void EXIT(int code)
{
#ifdef INTERACTIVE
	char buffer;
	printf("[Hit enter to continue]");
	fflush(stdout);
	read(0,&buffer,sizeof(char));
#endif
	exit(code);
}

/* ReadFile
 * 	Reads a file into a buffer that is allocates then returns the
 * 	size of the buffer that it allocated.
 * Input
 * 	filename : name of the file to be read in
 * 	buffer   : pointer that will point to the buffer
 * Returns
 * 	size of buffer allocated
 * Note:
 * 	This function allocates memory that should be freed after you
 * 	are done with it.
 */
size_t ReadFile(const char *filename, char **buffer)
{
    int fd;
    struct stat statbuff;
    size_t size;

    if( filename==NULL )
    {
	fprintf(stderr, "NULL filename in ReadFile\n");
	EXIT(-1);
    }

    fd = open(filename,O_RDONLY|O_BINARY);
    if( fd<0 )
    {
	fprintf(stderr,"Unable to open file \"%s\"\n", filename);
	perror(filename);
	EXIT(-1);
    }

    if( fstat(fd,&statbuff)<0 )
    {
	fprintf(stderr,"\nUnable to access file \"%s\"\n", filename);
	perror(filename);
	EXIT(-1);
    }
    size=statbuff.st_size;
    *buffer = (char*)malloc(size);

    read(fd,*buffer,size);

    return size;
}

/* VersionCheck
 * 	Try to determine what version of the Petz program a pet file is for.
 * Input
 * 	buffer : a pointer to a buffer that has a pet file in it
 * 	size   : the size of that buffer
 * Returns
 * 	3 if the buffer contains a Petz 3 pet
 * 	2 if the buffer contains a Petz 2 pet (not well tested)
 * 	-1 if the version can't be determined.
 */
int VersionCheck(char *buffer, size_t size)
{
    int i=0;

    for(i=0;i<size-7;i++)
    {
	if( !strncmp(buffer+i,"PETZIII",7) )
	    return 3;
	if( !strncmp(buffer+i,"PETZ II",7) )
	    return 2;
    }

    return -1;
}

/* TestBMP
 * 	Runs a couple tests on the bitmap that is to become the wallpaper
 * 	to determine of it is the correct type for the Petz player.
 * Input
 * 	buffer : a buffer that contains a bitmap file
 * 	size   : the size of the buffer
 * Returns
 * 	0 if the bitmap passes all the tests
 * 	-1 if the buffer is smaller than a bitmap header
 * 	-2 if the bitmap doesn't start with "BM"
 * 	-3 if the bitmap is not 8-bit
 */
int TestBMP(char *buffer, size_t size)
{
    /* check to see that the file is long enough to have a bitmap header */
    if( size<54 )
	return -1;
    /* check that it starts with th magical BM string */
    if( buffer[0]!='B' || buffer[1]!='M' )
	return -2;
    /* check to see that it is the correct pixel depth */
    if( buffer[28]!=8 )
	return -3;

    return 0;
}

/* FixExtension
 * 	Check to see that a filename has the expected extension.  If it
 * 	doesn't ask the user if they want to correct extension appended
 * 	to the filename.
 * Input
 * 	filename : the filename to be checked
 * 	ext      : the extension that is expected
 * Returns
 *	0 is no modifications were made to the filename
 *	1 is the filename was altered
 */
int FixExtension(char *filename,char *ext)
{
    int StartExt=0;
    char newname[512];

    StartExt=strlen(filename)-strlen(ext);
    /* check to see if the extension is ok */
    if(!strcmp(filename+StartExt,ext))
	return 0;

    if( filename[StartExt]=='.' )
    {
	strcpy(newname, filename);
	strcpy(newname+StartExt,ext);
    }
    else
	sprintf(newname,"%s%s",filename,ext);

#if 0
    printf("\n%s doesn't have the expected extension of %s.\n", filename, ext);
    printf("Would you like me to change it to %s? [y/n]: ", newname);
    scanf("%s",answer);
    if( tolower(answer[0])!='y' )
	return 0;
#endif
    strcpy(filename,newname);
    return 1;
}

/* isType
 * 	Check to see if a file has the specified extension.
 * Input
 * 	filename : name of file to be checked
 * 	ext	 : extension to test for
 * Output
 * 	none
 * Returns
 * 	1 if filename ends with ext
 * 	0 otherwise
 */
int isType(char *filename, char *ext)
{
    int i,j;

    i=strlen(filename)-strlen(ext);
    j=0;

    while( filename[i] && ext[j] )
    {
	if( tolower(filename[i++])!=tolower(ext[j++]) )
	    return 0;
    }
    return 1;
}

/* HTMLify
 * 	Converts filenames into a sufficiently encoded format for inclusion
 * 	in a URL.
 * Input
 * 	rawname   : buffer containing the unencoded string
 * 	fixedname : buffer that will receive the encoded string
 * Returns
 * 	number of characters that needed to be modified.
 */
#ifdef MAKEHTML
int HTMLify(char *rawname, char *fixedname)
{
    size_t rawlength=0;
    int changes=0;
    int i=0;
    int offset=0;
    char temp[512];

    strcpy(temp,rawname);
    i=strlen(rawname);
    while( i>0 && rawname[i]!='/' && rawname[i]!='\\') i--;
    if( rawname[i]=='/' || rawname[i]=='\\' )
	strcpy(temp,rawname+i+1);
    else
	strcpy(temp,rawname+i);

    rawlength=strlen(temp);
    for(i=0;i<rawlength+1;i++)
    {
	if( isalnum(temp[i]) || temp[i]=='.' || temp[i]=='\0' )
	    fixedname[offset++]=temp[i];
	else
	{
	    offset+=sprintf(fixedname+offset,"%%%02X",temp[i]);
	    changes++;
	}
    }

    return changes;
}
#endif

/* PlyGen
 * 	Make a ply file out of a petfile and a bitmap
 * Input
 * 	PetFile : Name of the petfile to use
 * 	Bmpfile : Name of the image file to use
 * Output
 * 	PlyFile : Name of the PlyFile written
 * 	Htmlfile : Name of the html file written
 * Returns
 * 	1 if files were successfully created
 * 	0 otherwise
 */
int PlyGen(char *PetFileName, char *PicFileName,
	   char *PlyFileName, char *HtmlFileName)
{
    size_t PetLen = 0,
	   PicLen = 0,
	   PlyLen = 0;
    char *PetFileBuff=NULL,
	 *PicFileBuff=NULL,
	 *PlyFileBuff=NULL;
#ifdef MAKEHTML
    char HtmlFileBuff[1024],
	 FixedPlyFileName[512];
    int  changed=0;
#endif
    static char LastPic[512];
    int  version=0,
	 code=0;
    char answer[1024];
    char PlySep[8], PlyEOF[9];
    char petname[100];
    char *curr=NULL;
    char header[1000];
    int  headlen = 0 ;
    int  misc = 2;
    int  fd=0;

    /* figure out whet the output file will be called */
    strcpy(PlyFileName,PetFileName);
    FixExtension(PlyFileName,".ply");

#ifdef MAKEHTML
    strcpy(HtmlFileName,PlyFileName);
    strcpy(HtmlFileName+strlen(HtmlFileName)-4,".html");
#endif

    /* read in the pet and check it's version */
    PetLen = ReadFile(PetFileName,&PetFileBuff);
    version = VersionCheck(PetFileBuff,PetLen);

    if( version==3 )
    {
#ifdef INTERACTIVE
	printf("%s appears to be a Petz 3 file, good.\n", PetFileName);
#endif
    }
    else
    {
	fprintf(stderr,"\"%s\" does not appear to be a Petz 3 file.\n",PetFileName);
#ifdef INTERACTIVE
	printf("\tIf you continue the ply will most likely not work.\n");
	printf("Create %s anyway? [y/n]: ", PlyFileName);
	scanf("%s",answer);
	if( tolower(answer[0])!='y' )
	{
	    printf("\n");
	    return 0;
	}
	printf("\nOk, have it your way, but don't say I didn't warn you!\n\n");
#endif
    }

    PicLen = ReadFile(PicFileName,&PicFileBuff);
    code = TestBMP(PicFileBuff,PicLen);
    if( code<0 )
    {
	if( strncmp(LastPic,PicFileName,511) )
	{
	    fprintf(stderr,"\"%s\" doesn't appear to be the correct type of bitmap file.\n", PicFileName);
#ifdef INTERACTIVE
	    printf("If you continue the ply may not work or might have a ");
	    printf("different\nwallpaper than you expect.\n");
	    printf("Create %s anyway? [y/n]: ", PlyFileName);
	    scanf("%s",answer);
	    if( tolower(answer[0])!='y' )
	    {
		printf("\n");
		return 0;
	    }
	    printf("\nOk, have it your way, but don't say I didn't warn you!\n\n");
	}
	else
	{
	    printf("%s might not work, but you said to use it.\n", PicFileName);
#endif
	}
    }
    else
    {
#ifdef INTERACTIVE
	printf("%s looks like an ok bitmap file, good.\n", PicFileName);
#endif
    }
    strncpy(LastPic,PicFileName,511);

    /* create necessary separators */
    PlySep[0] = '\0';
    strcpy(PlySep+1,"plySep");
    PlyEOF[0] = '\0';
    strcpy(PlyEOF+1,"plyEOF");

    /* figure out the pet's name */
    strcpy(petname,PetFileBuff+2);
#ifdef INTERACTIVE
    printf("Starting to make %s from %s and %s... ",
	    PlyFileName, PetFileName, PicFileName);
    fflush(stdout);
#endif

    /* build the header */
    curr=header;
    *curr = '\0';	/* first char seems to be a null */

    curr=header+1;
    strcpy(curr, "plyHed");	/* add in the PlyHed delimiter */

    /* length of ply */
    curr = header+8;
    PlyLen = PetLen+PicLen+strlen(petname)+39+8+16;
    memcpy(curr, &PlyLen,sizeof(long));
    PlyFileBuff = (char*)malloc(PlyLen);

    /* unknown bytes (always seems to be 0x02 */
    curr = header+12;
    misc=2;
    memcpy(curr, &misc,sizeof(long));

    curr = header+16;
    strcpy(curr,petname);
    curr += strlen(petname);
    strcpy(curr,".pet");

    /* length of pet file */
    curr += 5;
    memcpy(curr, &PetLen,sizeof(long));

    /* wall paper delimiter */
    curr += 4;
    strcpy(curr,"wallpaper");

    /* length of bg file */
    curr += strlen("wallpaper")+1;
    memcpy(curr, &PicLen,sizeof(long));
    curr += 4;
    headlen = curr-header;

    /* assemble the ply file */
    curr = PlyFileBuff;
    memcpy(curr, header, headlen);
    curr += headlen;

    memcpy(curr,PetFileBuff,PetLen);
    curr += PetLen;

    memcpy(curr,PlySep,8);
    curr += 8;

    memcpy(curr,PicFileBuff,PicLen);
    curr += PicLen;

    memcpy(curr,PlySep,8);
    curr += 8;
    memcpy(curr,PlyEOF,8);

    fd = open(PlyFileName,O_WRONLY|O_CREAT|O_TRUNC|O_BINARY,S_IREAD|S_IWRITE);
    if( fd<0 )
    {
	fprintf(stderr,"\nUnable to create output file  \"%s\".\n", PlyFileName);
	perror(PlyFileName);
	EXIT(-1);
    }
    if( write(fd,PlyFileBuff,PlyLen) != PlyLen)
    {
	fprintf(stderr,"\nProblem while writing %s.\nPerhaps your drive is full\n",
		PlyFileName);
    }
    close(fd);
    free(PetFileBuff);
    free(PicFileBuff);
    free(PlyFileBuff);

#ifdef MAKEHTML
    changed = HTMLify(PlyFileName,FixedPlyFileName);
    sprintf(HtmlFileBuff,"<EMBED SRC=\"%s\" TYPE=\"application/x-PFM plugin\" WIDTH=\"500\" HEIGHT=\"500\" UNITS=\"pixels\">\n", FixedPlyFileName);
    fd=open(HtmlFileName,O_WRONLY|O_CREAT|O_TRUNC|O_BINARY,S_IREAD|S_IWRITE);
    if( fd<0 )
    {
	fprintf(stderr,"\nUnable to create output file  \"%s\".\n", HtmlFileName);
	perror(HtmlFileName);
	EXIT(-1);
    }
    write(fd,HtmlFileBuff,strlen(HtmlFileBuff));
    close(fd);
#endif

#ifdef INTERACTIVE
    printf("Done.\n\tWrote %s and %s.\n\n", PlyFileName, HtmlFileName);
#endif
    return 1;
}

/* Main
 * 	This is where it all begins
 * Input
 * 	argc : the number of command line arguments
 * 	argv : array of pointers to the arguments
 * Returns
 * 	0 on completion
 */
int main(int argc, char *argv[])
{
    char PetFileName[512],
	 PicFileName[512],
	 PlyFileName[512],
	 HtmlFileName[512],
	 *ConfFileBuff=NULL;
    int i=0;
    int count=0;
    int ConfLen=0;
    FILE *fp=NULL;
    char answer[512],
         *curr=NULL;
    char *config=NULL;
    struct stat statBuff;
    int beeped=0;

#ifdef INTERACTIVE
    printf("Welcome to Fraggle's Pet2Ply converter.\n\n");
    printf("    pet2ply version %s, Copyright (C) 1999 Bryan Franklin\n", VERSION);
    printf("    pet2ply comes with ABSOLUTELY NO WARRANTY; for details run 'pet2ply -w'.\n");
    printf("    This is free software, and you are welcome to redistribute it under\n");
    printf("    certain conditions; run 'pet2ply -c' for details.\n\n");
#endif

    /* compute the config file's name */
#ifdef WIN32
    config=(char*)malloc(strlen(argv[0])+1);
    strcpy(config,argv[0]);
    i=strlen(config);
    while( config[i]!='/' && config[i]!='\\') i--;
    strcpy(config+i+1,CONFIGFILE);
#elif defined(UNIX)
    /* I realize I should probably have it look for the config file in $HOME
     * But I'm lazy.  If you really want that behavior send me a patch */
    config=(char*)malloc(strlen(CONFIGFILE)+1);
    strcpy(config,CONFIGFILE);
#endif

    /* check to see if they just double clicked */
#ifdef INTERACTIVE
    if( argc<=1 )
    {
	printf("\n\nYou are trying to run pet2ply without any input.\n\n");
	printf("This is not exactly the easiest way to run pet2ply.  I suggest you hit Ctrl+C\n");
	printf("and take a look at readme.txt for instructions on how to use the pet2ply\n");
	printf("converter.\n\nThere is a much easier way to use pet2ply.\n\n\a");
	beeped=1;
    }

    /* configure pet2ply */
    if( argc==2 && isType(argv[1],".bmp") )
    {
	printf("Are you sure you want to set the default wallpaper image to\n\"%s\"? [y/n] ", argv[1]);
	fflush(stdout);
	scanf("%s",answer);
	if( tolower(answer[0])!='y' )
	    EXIT(0);
#endif

	fp=fopen(config,"wt");
	if( fp==NULL )
	{
	    fprintf(stderr,"Unable to open config file %s\n", config);
	    perror("fopen");
	    EXIT(-1);
	}
	fprintf(fp,"wallpaper=%s\n",argv[1]);
	fclose(fp);
	EXIT(0);
    }

#ifndef INTERACTIVE
    if( argc>1 && argc<3 )
	fprintf(stderr, "Usage:\n\t%s [infile.bmp] infile.pet [...]\n\n",argv[0]);
#endif

    for(i=1; i<argc; i++)
    {
        if( !strcmp("-w",argv[i]) )
        {
            printf("    This program is distributed in the hope that it will be useful,\n");
            printf("    but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
            printf("    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
            printf("    GNU General Public License for more details.\n");
            EXIT(0);

        }
        else if( !strcmp("-c",argv[i]) )
        {
            printf("    This program is free software; you can redistribute it and/or modify\n");
            printf("    it under the terms of the GNU General Public License as published by\n");
            printf("    the Free Software Foundation; either version 2 of the License, or\n");
            printf("    (at your option) any later version.\n");
            EXIT(0);
        }
    }


    /* scan command line arguments for necessary parameters */
    PetFileName[0]='\0';
    PicFileName[0]='\0';
    for( i=1; i<argc; i++)
    {
	if( PetFileName[0]=='\0' && isType(argv[i],".pet") )
	    strncpy(PetFileName,argv[i],511);
	if( PicFileName[0]=='\0' && isType(argv[i],".bmp") )
	    strncpy(PicFileName,argv[i],511);
    }

    /* get the name of the wallpaper image to use */
    if( PicFileName[0]=='\0' )
    {
	if( stat(config,&statBuff)==0 )
	{
	    ConfLen=ReadFile(config,&ConfFileBuff);
	    if( ConfLen>0 )
	    {
		curr=strtok(ConfFileBuff,"=\n");
		while( curr!=NULL )
		{
		    if( !strcmp(curr, "wallpaper") )
		    {
			curr=strtok(NULL,"\n\r");
			strcpy(PicFileName,curr);
		    }
		    curr=strtok(NULL," \t\n");
		}
	    }
	    free(ConfFileBuff);
#ifdef INTERACTIVE
	    printf("Using \"%s\" for the wallpaper\n\n", PicFileName);
#endif
	}
	else
	{
#ifdef INTERACTIVE
	    if( !beeped )
	    {
		printf("You do not have a wallpaper image configured.\n\a\n");
		printf("Pet2ply is much easier to use if you configure a wallpaper image.\n");
		printf("Please refer to readme.txt to see how to configure the wallpaper.\n\n");
		beeped=1;
	    }
	    printf("Please enter the name of the bitmap file you would like to use for a\n");
	    printf("wallpaper image.  Be sure to include the extension (.bmp) and the full\n");
	    printf("path to the file, if necessary.\n\n");
	    printf("Background image file: ");
	    fflush(stdout);
	    i=0;
	    while( i<sizeof(answer) && read(0,answer+i,1) && answer[i]!='\n' ) i++;
	    answer[i]='\0';
	    strcpy(PicFileName,answer);
#else
	    fprintf(stderr, "Missing wallpaper file on command line\n");
#endif
	}
    }
    free(config);
    FixExtension(PicFileName,".bmp");

    /* get the name of the pet file to use */
    if( PetFileName[0]=='\0' )
    {
#ifdef INTERACTIVE
	printf("\nPlease enter the name of the pet file you want to convert to ply format.\n");
	printf("Be sure to include the extension (.pet) and the full path to the file,\n");
	printf("if necessary.\n\n");
	printf("Pet file: ");
	fflush(stdout);
	i=0;
	while( i<sizeof(answer) && read(0,answer+i,1) && answer[i]!='\n' ) i++;
	answer[i]='\0';
	strcpy(PetFileName,answer);
	count+=PlyGen(PetFileName,PicFileName,PlyFileName,HtmlFileName);
#else
	fprintf(stderr, "Missing pet file on command line\n");
#endif
    }
    FixExtension(PetFileName,".pet");

    /* loop through command line and make ply file form the pet files */
    for( i=1; i<argc; i++)
    {
	if( isType(argv[i],".pet") )
	{
	    strncpy(PetFileName,argv[i],511);
	    count+=PlyGen(PetFileName,PicFileName,PlyFileName,HtmlFileName);
	}
	else if( isType(argv[i],".bmp") )
	    strncpy(PicFileName,argv[i],511);
	else
	    fprintf(stderr,"Unknown filename \"%s\" on command line\n",argv[i]);

    }

#ifdef INTERACTIVE
#ifdef MAKEHTML
    printf("Wrote %i ply files along with their html files.\n", count);
#else
    printf("Wrote %i ply files.\n", count);
#endif
    printf("Thank you for using pet2ply and may the source be with you.\n");
#endif

    EXIT(0);
    return 0;
}
