/*
 * pet2bmp is a utility for extracting bitmaps encoded in a pet files.
 * Copyright (C) 1999 Bryan Franklin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/* if you would like to compile on a Unix machine instead
 * of a windows machine simply comment out the WIN32 line
 * and uncomment the UNIX line.
 */
#define WIN32
/*#define UNIX*/

/* If don't want pet2bmp to ever ask you a question
 * then comment out the following line and recompile.
 * This will not only make pet2bmp a lot smaller, but
 * will make it more suitable for the back end of a web-page.
 */
#define INTERACTIVE

/* Everything needs config file, right? */
#ifdef WIN32
#define CONFIGFILE	"config.ini"
#elif defined(UNIX)
#define CONFIGFILE	".pet2bmprc"
#endif

#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>
#ifdef UNIX
#include <unistd.h>
#define O_BINARY 0	/* ahh, the joys of porting to windows */
#define S_IREAD S_IRUSR
#define S_IWRITE S_IWUSR
#endif

#define VERSION "1.1"

#define LOW_LEFT (54+1024)
#define LOW_RIGHT (54+1024+107)
#define UP_LEFT (54+1024+79*108)
#define UP_RIGHT (54+1024+79*108+107)

struct color
{
    char name[255];
    char triple[255];
    char hex[255];
};

/* EXIT
 * 	Allow the user to hit enter before the program terminates
 * Input
 * 	code : exit code that the program returns to the OS
 * Output
 * 	none, program terminates in this function
 * Returns
 * 	none, see output
 */
void EXIT(int code)
{
#ifdef INTERACTIVE
    char buffer[100];
    printf("[Hit enter to continue]");
    fflush(stdout);
    read(0,buffer,1);
#endif
    exit(code);
}

/* FindArray
 * 	searches an array for the contents of another array, essentially
 * 	a substring search.
 * Input
 * 	a1    : array to search for
 * 	size1 : size of a1 (in elements)
 * 	a2    : array to look in for a1
 * 	size2 : size of a2 (in elements)
 * 	start : element of a2 to start the search at
 * Output
 * 	none
 * Returns
 * 	The offset in a2 where the array a1 was found, -1 if a1 was not found.
 */
int FindArray(unsigned char *a1, size_t size1,
	      unsigned char *a2, size_t size2,
	      int start)
{
    int offset=0;
    int done=0;
    int i=0;

    if( size2<size1 )
	return -1;

    for( offset=start; !done && offset<size2-size1; offset++ )
    {
	done=1;
	for(i=0; i<size1; i++)
	{
	    if( a1[i] != a2[offset+i] )
	    {
		done=0;
		break;
	    }
	}
	if( done ) return offset;
    }

    return -1;
}

/* ExtractPic
 * 	Attempts to find a bitmap in a buffer and repairs the headers of that
 * 	bitmap.
 * Input
 * 	buffer : buffer that is searched for a bitmap
 *	size   : size of buffer
 *	offset : where in the buffer to start the search
 * Output
 * 	*image : pointer that is set to the beginning of the image found
 * Returns
 * 	Location in buffer where image was found.
 * Notes
 * 	This function modifies the contents of buffer.
 */
int ExtractPic( unsigned char **image, unsigned char *buffer,
						    size_t size, size_t offset)
{
    int start=0;
    unsigned char header[]={66,77,54,238,2,0,0,0, 0,0,54,4,0,0,40,0,
		    0,0,108,0,0,0,80,0, 0,0,1,0,8,0,0,0,
		    0,0,0,238,2,0,0,0, 0,0,0,0,0,0,0,0,
		    0,0,0,0,0,0};
    char Sizes[]={108,0,0,0,80,0,0,0};

    /* find start of image section */
    start = FindArray(Sizes,8, buffer,size, offset+18);
    if( start<0 )
	return -1;
    start -= 18;

    (*image)=buffer+start;
    if( *image>buffer+size || *image<buffer )
    {
	fprintf(stderr,"image pointer out of bounds\n");
	return -1;
    }

    memcpy(*image,header,14);

    return *image-buffer;
}

/* TestPic
 * 	Runs some tests on the image that was found to see that it is a valid
 * 	bitmap.
 * Input
 * 	image : buffer containing the image to be tested
 * Output
 * 	none
 * Returns
 * 	flag to indicate is it is a valid image
 *	1	valid
 *	0	invalid
 */
int TestPic(unsigned char *image)
{
    unsigned long corners=0;
    unsigned char red=0, green=0, blue=0;
    char answer[100];

    if( image[LOW_LEFT]==image[LOW_RIGHT] &&
	image[LOW_RIGHT]==image[UP_LEFT] &&
	image[UP_LEFT]==image[UP_RIGHT] )
	corners = image[UP_LEFT];

    red = image[54+4*corners];
    green = image[54+4*corners+1];
    blue = image[54+4*corners+2];

    if( red==0xff && green==0x0 && blue==red )
	return 1;

#ifdef INTERACTIVE
    printf("\nThe image as found does not appear to be valid.\n");
    printf("Would you like to write it anyway? [y/n]: ");

    scanf("%s",answer);
    if( tolower(answer[0])=='y' )
    {
	printf("\nOk, have it your way, but don't expect a pretty picture (if it works at all).\n");
	return 1;
    }

    printf("\nImage will not be written.\n");
#else
    fprintf(stderr,"Image doesn't appear to be valid, writing anyway\n");
    return 1;	/* might as well write it if we can't ask */
#endif

    return 0;
}

/* GetImageStats
 * 	Reads some of the size and pixel depth information from an image
 * Input
 * 	image : buffer that contains the image
 * Output
 * 	xres   : width of the image (in pixels)
 * 	yres   : height of the image (in pixels)
 * 	colors : number of possible colors in this image type
 * 	depth  : number of bits per pixel per plane
 * Returns
 * 	nothing
 * Note
 * 	This function requires that the image buffer be at least 28 bytes long.
 * 	A real bitmap header is generally up to 54 bytes long.
 */
void GetImageStats(char *image, int *xres, int *yres, int *colors,int *depth)
{
    *xres=image[18]+(image[19]<<8)+(image[20]<<16)+(image[21]<<24);
    *yres=image[22]+(image[23]<<8)+(image[24]<<16)+(image[25]<<24);
    *depth=image[28];
    *colors=1<<(*depth);
}

/* ProcessFile
 *      extract all of the images from a file
 * Input
 *      filename : name of the file to extract from
 *      offset   : where in the file to start looking
 *      red      : red component of bg color
 *      green    : green "
 *      blue     : blue "
 * Output
 *      none, but it does potentially write files.
 * Returns
 *      Number of images found
 */
int ProcessFile(char *filename, long offset,unsigned char red, unsigned char green, unsigned char blue)
{
    struct stat stats;
    size_t size;
    unsigned char *buffer=NULL, *image1=NULL;
    int picnum=0;
    int fd=1;
    char *output=NULL, *basename=NULL;

#ifdef INTERACTIVE
    printf("\nExamining %s\n", filename);
#endif
    output=(char*)malloc(strlen(filename)+1+4);
    if( output==NULL )
    {
	fprintf(stderr,"malloc: out of memory\n");
	EXIT(-1);
    }

    strcpy(output,filename);
    basename=(char*)malloc(strlen(output)+1);
    if( basename==NULL )
    {
	fprintf(stderr,"malloc: out of memory\n");
	EXIT(-1);
    }

    strcpy(basename,filename);
    if( !strcmp(".pet",output+strlen(output)-4) ||
	!strcmp(".PET",output+strlen(output)-4))
    {
	strcpy(output+strlen(output)-4,".bmp");
	strcpy(basename+strlen(basename)-4,"");
    }
    else
    {
	fprintf(stderr,"\n%s does not end in .pet or .PET, simply appending .bmp to output file\n", output);
	strcpy(output+strlen(output),".bmp");
    }

    /* allocate memory for the file */
    if( stat(filename,&stats) )
    {
	perror(filename);
	EXIT(-2);
    }
    size=stats.st_size;
    buffer=(char*)malloc(size);
    if( buffer==NULL )
    {
	fprintf(stderr,"malloc: unable to allocate %i bytes of memory\n", size);
	EXIT(-3);
    }

    /* read in the file */
    fd=open(filename,O_RDONLY|O_BINARY);
    if( fd<0 )
    {
	perror("open");
	EXIT(-4);
    }
    read(fd,buffer,size);
    close(fd);

    /* extract the picture */
    picnum=0;
    while( (offset=ExtractPic(&image1,buffer,size,offset))>=0
							&& TestPic(image1) )
    {
	int xres,yres,colors,depth,wrote;
	if( picnum>0 )
	    sprintf(output,"%s%i.bmp",basename,picnum);
	GetImageStats(image1,&xres,&yres,&colors,&depth);
#ifdef INTERACTIVE
	printf("  Image found at offset %li: ",offset);
	printf("image dimensions are %dx%dx%d\n",xres,yres,colors);
	printf("  Saving as %s: ", output);
#endif
	offset++;

	/* set the background color */
	image1[54+4*253+2]=red;
	image1[54+4*253+1]=green;
	image1[54+4*253+0]=blue;

	fd=open(output,O_WRONLY|O_CREAT|O_TRUNC|O_BINARY,S_IREAD|S_IWRITE);
	wrote=write(fd,image1,xres*yres*(depth/8)+1024+54);
	close(fd);
	picnum++;
#ifdef INTERACTIVE
	printf("wrote %d bytes.\n",wrote);
#endif
    }

#ifdef INTERACTIVE
    if( picnum==0 )
    {
	fprintf(stderr, "Unable to find valid pictures in %s\n\n", filename);
	EXIT(-1);
    }
    else
	printf("Found %i image(s) in %s\n\n", picnum, filename);
#endif

    free(buffer);
    free(output);
    free(basename);
    return picnum;
}


/* ColorMenu
 * 	Display the color menu and allow return selection
 * Input
 * 	none
 * Output
 * 	BgColor : Background color
 * Return
 * 	nothing
 */
#ifdef INTERACTIVE
void ColorMenu(char *BgColor)
{
    int i;
    int TotColors=12;
    char answer[256];
    int choice=0;
    struct color colors[] = {
			{"Black ",	"0,0,0      ",	"#000000" },
			{"Red   ",	"255,0,0    ",	"#ff0000" },
			{"Orange",	"255,128,0  ",  "#ff8000" },
			{"Brown ",	"128,96,64  ",	"#806040" },
			{"Yellow",	"255,255,0  ",	"#ffff00" },
			{"Green ",	"0,255,0    ",	"#00ff00" },
			{"Cyan  ",	"0,255,255  ",	"#00ffff" },
			{"Blue  ",	"0,0,255    ",	"#0000ff" },
			{"Purple",	"192,0,255  ",	"#c000ff" },
			{"Magenta",	"255,0,255  ",	"#00ff00" },
			{"Pink  ",	"255,128,192",	"#ff80c0" },
			{"White ",	"255,255,255",	"#ffffff" }
		      };

    do
    {
	printf("\n\tBackground Color\n\t----------------\n");
	for(i=0; i<TotColors; i++)
	{
	    printf("\t%i) %s\t%s\t%s\n", i+1, colors[i].name,
		    colors[i].triple, colors[i].hex);
	}
	printf("\tC) Custom color\n");
	printf("\tQ) Back to Main menu\n");
	printf("\n\tCurrent color is %s\n", BgColor);
	printf("\tWhich would you like? ");
	scanf("%s",answer);
	if( tolower(answer[0]) == 'c' )
	{
	    printf("\n\tEnter the custom color as either a triple or in hex: ");
	    scanf("%s",BgColor);
	}
	else if( isdigit(answer[0]) )
	{
	    choice = atoi(answer);
	    if( choice<=TotColors )
		strcpy(BgColor,colors[choice-1].triple);
	    else
		printf("\n\t%i is out of range, please try again.\n", choice);
	}
    } while( tolower(answer[0]) != 'q' && tolower(answer[0]) != 'b' );
    strtok(BgColor," ");	/* get rid of those pesky spaces */
}

/* MakeConfig
 * 	Ask the user some questions and create a config file
 * Input
 * 	offset  : Pointer to offset
 * Output
 * 	BgColor : 'name' of color to use for background
 * 	*offset : Value of offset
 * Return
 * 	1 if file written
 * 	0 if nothing done
 */
int MakeConfig(char *filename, char *BgColor, long *offset)
{
    char answer[256];
    FILE *fp=NULL;
    int unsaved=1;
    char *OldColor=NULL;
    int OldOffset=0;

    printf("Would you like to create a new configuration file? [y/n] ");
    fflush(stdout);
    scanf("%s",answer);
    if( tolower(answer[0]) == 'n' )
    {
	strcpy(BgColor,"#ff00ff");
	*offset = 1;
	printf("Ok, defaulting to a background of %s and an offset of %i.\n",
		BgColor, *offset);
	return 0;
    }

    strcpy(BgColor,"255,0,255");
    *offset = 1;	/* no need to find a picture if given a bitmap */

    while( tolower(answer[0]) != 'q' )
    {
	printf("\n\tMain Menu\n\t---------\n");
	printf("\t1) Background color [%s]\n", BgColor);
	printf("\t2) Starting offset  [%li]\n", *offset);
	printf("\tS) Save %s\n",unsaved?"Needed":"");
	printf("\tQ) Quit\n");
	printf("\n\tWhich option would you like to configure? ");
	fflush(stdout);
	scanf("%s",answer);

	switch( answer[0] )
	{
	    case '1':
		OldColor=malloc(strlen(BgColor)+1);
		strcpy(OldColor,BgColor);
		ColorMenu(BgColor);
		if( strcmp(OldColor,BgColor) )
		    unsaved=1;
		if( OldColor )
		    free(OldColor);
		break;
	    case '2':
		OldOffset=*offset;
		printf("\n\tOffset? ");
		fflush(stdout);
		scanf("%s",answer);
		*offset = atoi(answer);
		if( *offset!=OldOffset )
		    unsaved=1;
		break;
	    case 's':
	    case 'S':
		fp = fopen(filename,"wt");
		if( fp == NULL )
		{
		    fprintf(stderr,"Unable to create file %s.\n", filename);
		    perror("fopen");
		    return 0;
		}
		fprintf(fp,"background\t%s\n", BgColor);
		fprintf(fp,"offset\t\t%li\n", *offset);
		fclose(fp);
		printf("\tSaved...\n");
		unsaved=0;
		break;
	    case 'q':
	    case 'Q':
		if( unsaved )
		{
		    printf("\n\tAre you sure you want to quit without saving? [y/n] ");
		    scanf("%s",answer);
		    if( tolower(answer[0])=='y' )
		    {
			answer[0]='q';
			printf("\tQuiting\n\n");
		    }
		    else
		    {
			printf("\n\tI didn't think that's what you really wanted. :)\n");
		    }
		}
		break;
	    default:
		printf("Please enter 1, 2, S or Q!\n");
	}
    }

    return 1;
}
#endif /* INTERACTIVE */


/* main
 * 	the magical starting point
 * Input
 * 	argc : number of command line arguments
 * 	argv : array of pointers to the arguments
 * Output
 * 	none
 * Return
 *	generally doesn't (EXIT(0) is used instead)
 */
int main(int argc, char *argv[])
{
    long offset=0,
         color1=0,color2=0,color3=0;
    int i, total=0;
    unsigned char red=255, green=0, blue=255;
    char BgColor[256];
    char buffer[512];
    char *config=NULL;
    FILE *fp=NULL;

#ifdef INTERACTIVE
    printf("Welcome to Fraggle's Pet2Bmp converter.\n\n");
    printf("    pet2bmp version %s, Copyright (C) 1999 Bryan Franklin\n", VERSION);
    printf("    pet2bmp comes with ABSOLUTELY NO WARRANTY; for details run 'pet2bmp -w'.\n");
    printf("    This is free software, and you are welcome to redistribute it under\n");
    printf("    certain conditions; run 'pet2bmp -c' for details.\n\n");
#endif

    /* parse command line options */
#ifndef INTERACTIVE
    if( argc<=1 )
    {
	fprintf(stderr,"Usage:\n\t%s petfile.pet [...]\n\n", argv[0]);
    }
#endif

    for(i=1; i<argc; i++)
    {
	if( !strcmp("-w",argv[i]) )
	{
	    printf("    This program is distributed in the hope that it will be useful,\n");
	    printf("    but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
	    printf("    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
	    printf("    GNU General Public License for more details.\n");
	    EXIT(0);

	}
	else if( !strcmp("-c",argv[i]) )
	{
	    printf("    This program is free software; you can redistribute it and/or modify\n");
	    printf("    it under the terms of the GNU General Public License as published by\n");
	    printf("    the Free Software Foundation; either version 2 of the License, or\n");
	    printf("    (at your option) any later version.\n");
	    EXIT(0);
	}
    }


    /* compute the config file's name */
#ifdef WIN32
    config=(char*)malloc(strlen(argv[0])+1);
    strcpy(config,argv[0]);
    i=strlen(config);
    while( config[i]!='/' && config[i]!='\\') i--;
    strcpy(config+i+1,CONFIGFILE);
#elif defined(UNIX)
    /* I realize I should probably have it look for the config file in $HOME
     * But I'm lazy.  If you really want that behavior send me a patch */
    config=(char*)malloc(strlen(CONFIGFILE)+1);
    strcpy(config,CONFIGFILE);
#endif

    /* now read the file */
    printf("Opening config file \"%s\"\n", config);
    fp = fopen(config,"rt");
#ifdef INTERACTIVE
    if( argc<=1 )
    {
	if( MakeConfig(config,BgColor,&offset) )
	{
	    printf("\n\nNow you should be able to simply drag a pet file onto the pet2bmp icon and a\n");
	    printf("bitmap file should appear in the directory the pet file was in.\n\n");
	    EXIT(0);
	}
    }
    else if( fp == NULL )
    {
	printf("\nUnable to open config file, %s.\n", config);
	MakeConfig(config,BgColor,&offset);
	fp = fopen(config,"rt");
    }
#endif

    if( fp != NULL )
    {
	while( !feof(fp) )
	{
    	    fscanf(fp,"%s ", buffer);
	    if( !strcmp("background",buffer) )
	    {
		fscanf(fp,"%s\n",buffer);
    		if( buffer[0]=='#' )
    		{
		    sscanf(buffer+1,"%lx", &color1);
		    red   = (color1&0x00ff0000)>>16;
		    green = (color1&0x0000ff00)>>8;
		    blue  = (color1&0x000000ff);
    		}
    		else
    		{
		    sscanf(buffer,"%li,%li,%li", &color1,&color2,&color3);
		    red = color1;
		    green = color2;
		    blue = color3;
 		}
	    }
	    else if( !strcmp("offset",buffer) )
	    {
		fscanf(fp,"%s\n",buffer);
		offset = atoi(buffer);
	    }
	    else
	    	fprintf(stderr,"Unknown token %s in config file\n", buffer);
        }
	fclose(fp);
	free(config);

#ifdef INTERACTIVE
	printf("\nUsing background color #%02x%02x%02x ",red,green,blue);
	printf("and offset %li.\n", offset);
#endif
    }

    /* process all of the files on the command line */
    if( argc > 1 )
    {
	for(i=1; i<argc; i++)
	    total += ProcessFile(argv[i],offset,red,green,blue);
    }
#ifdef INTERACTIVE
    else
    {
	printf("\nPlease enter the name of the file you want to extract a picture from\n");
	printf("(be sure to include the full path if necessary): ");
	fflush(stdout);
	i=0;
	while( i<sizeof(buffer) && read(0,buffer+i,1) && buffer[i]!='\n' ) i++;
	buffer[i]='\0';
	total += ProcessFile(buffer,offset,red,green,blue);
    }

    printf("Extraction complete, %i images found.\n\n", total);
#endif

    EXIT(0);

    return 0;
}
